// DOM elements
const elements = {
  statusBadge: document.getElementById('statusBadge'),
  
  // Views
  inactiveView: document.getElementById('inactiveView'),
  activeView: document.getElementById('activeView'),
  
  // Form inputs
  baseUrlInput: document.getElementById('baseUrl'),
  
  // Buttons
  startCapturingBtn: document.getElementById('startCapturingBtn'),
  downloadBtn: document.getElementById('downloadBtn'),
  deactivateBtn: document.getElementById('deactivateBtn'),
  
  // Info elements
  validationMessage: document.getElementById('validationMessage'),
  monitoringUrl: document.getElementById('monitoringUrl'),
  captureCount: document.getElementById('captureCount')
};

// State management
let state = {
  selectedBaseUrl: '',
  isActive: false
};

// Initialize popup
function initPopup() {
  // Load current state from storage
  chrome.storage.local.get(['isActive', 'selectedBaseUrl'], (data) => {
    state.isActive = data.isActive || false;
    state.selectedBaseUrl = data.selectedBaseUrl || '';
    
    updateUI();
  });
  
  // Set up event listeners
  setupEventListeners();
  
  // Update capture count periodically if active
  if (state.isActive) {
    updateCaptureCount();
    setInterval(updateCaptureCount, 2000);
  }
}

// Update UI based on current state
function updateUI() {
  // Update status badge
  elements.statusBadge.textContent = state.isActive ? 'Active' : 'Inactive';
  elements.statusBadge.classList.toggle('active', state.isActive);
  
  // Show the appropriate view
  elements.inactiveView.classList.toggle('active', !state.isActive);
  elements.activeView.classList.toggle('active', state.isActive);
  
  // Update input field if we have a saved URL
  if (!state.isActive && state.selectedBaseUrl) {
    elements.baseUrlInput.value = state.selectedBaseUrl;
  }
  
  if (state.isActive) {
    elements.monitoringUrl.textContent = state.selectedBaseUrl;
    updateCaptureCount();
  }
}

// Switch between views
function showView(viewName) {
  // Hide all views
  elements.inactiveView.classList.remove('active');
  elements.activeView.classList.remove('active');
  
  // Show the requested view
  elements[viewName + 'View'].classList.add('active');
}

// Update the capture count
function updateCaptureCount() {
  chrome.runtime.sendMessage({ action: 'getState' }, (response) => {
    if (response && response.capturedRequests !== undefined) {
      elements.captureCount.textContent = response.capturedRequests;
    }
  });
}

// Validate base URL format
function validateBaseUrl(url) {
  // Basic validation - ensure URL is not empty and follows a reasonable format
  if (!url) {
    return { valid: false, message: 'Please enter a base URL' };
  }
  
  // Remove protocol if present
  const cleanUrl = url.replace(/^(https?:\/\/)/, '');
  
  // Simple check for domain format
  const urlPattern = /^[a-zA-Z0-9][-a-zA-Z0-9.]*\.[a-zA-Z]{2,}(\/.*)?$/;
  if (!urlPattern.test(cleanUrl)) {
    return { valid: false, message: 'Please enter a valid URL' };
  }
  
  return { valid: true, cleanUrl };
}

// Set up event listeners
function setupEventListeners() {
  // Start capturing button click
  elements.startCapturingBtn.addEventListener('click', () => {
    const baseUrl = elements.baseUrlInput.value.trim();
    const validation = validateBaseUrl(baseUrl);
    
    if (!validation.valid) {
      showValidationMessage(validation.message, 'error');
      return;
    }
    
    // Clear any previous validation messages
    elements.validationMessage.textContent = '';
    
    // Update state
    state.selectedBaseUrl = baseUrl;
    state.isActive = true;
    
    // Send message to background script to activate
    chrome.runtime.sendMessage({
      action: 'activate',
      selectedBaseUrl: state.selectedBaseUrl
    }, (response) => {
      if (response && response.success) {
        // Update UI
        showView('active');
        elements.statusBadge.textContent = 'Active';
        elements.statusBadge.classList.add('active');
        elements.monitoringUrl.textContent = state.selectedBaseUrl;
        
        // Start updating capture count
        updateCaptureCount();
        setInterval(updateCaptureCount, 2000);
      } else {
        alert('Failed to activate extension. Please try again.');
      }
    });
  });
  
  // Download data button click
  elements.downloadBtn.addEventListener('click', () => {
    chrome.runtime.sendMessage({ action: 'downloadData' });
  });
  
  // Deactivate button click
  elements.deactivateBtn.addEventListener('click', () => {
    // Check if there's data to download
    chrome.runtime.sendMessage({ action: 'getState' }, (response) => {
      if (response && response.capturedRequests > 0) {
        // Prompt user to download data before stopping
        const downloadConfirm = confirm(`You have captured ${response.capturedRequests} requests. Would you like to download this data before stopping?`);
        
        if (downloadConfirm) {
          // Download data first
          chrome.runtime.sendMessage({ action: 'downloadData' }, () => {
            // Then deactivate
            deactivateExtension();
          });
        } else {
          // User chose not to download, just deactivate
          deactivateExtension();
        }
      } else {
        // No data to download, just deactivate
        deactivateExtension();
      }
    });
  });
  
  // Function to deactivate the extension
  function deactivateExtension() {
    // Send message to background script to deactivate
    chrome.runtime.sendMessage({ action: 'deactivate' }, (response) => {
      if (response && response.success) {
        // Update state
        state.isActive = false;
        
        // Update UI
        showView('inactive');
        elements.statusBadge.textContent = 'Inactive';
        elements.statusBadge.classList.remove('active');
      }
    });
  }
}

// Show validation message
function showValidationMessage(message, type) {
  elements.validationMessage.textContent = message;
  elements.validationMessage.className = 'message';
  
  if (type) {
    elements.validationMessage.classList.add(type);
  }
}

// Initialize the popup when the DOM is loaded
document.addEventListener('DOMContentLoaded', initPopup); 